/*  Firmware for 6-channel volume control, Model 465
/*
/*
/*  TDL Technology, Inc., Las Cruces, New Mexico USA
/*
/*  Written for type "PCM" PIC C compiler ver 3.00 or later
/*  by CCS, Inc.
/*
/*  Original written 02/18/2006
/*  Revised 7/21/2006 -- changed some of the I/O pins and added
/*	 a mute function.
/*  Revised 8/13/2006 -- implement mute function and changed version
/*	 name to firm465a.c
/*  Revised 1/8/2007 -- minor corrections and changed version name
/*	 to firm465b.c
/*  Revised 3/3/2007 -- added code to initialize stepper motor and
/*	 to store steppos in eeprom. Version still firm465b.c.
/*
/*  Set fuses for Crystal Oscillator, WDT off, Power Up Timer on,
/*  Code Protect off
/*
/*  source: firm465b.c
/*
/*	IR detector --> B7   pin 28
/*	Out voltage --> A1
/*	+5V ref     --> A3
/*	Mute relay  --> B0
/*	Mute LED    --> B1
/*	IR detect   --> B2
/*
*/
#include <16F873.h>
#include <stdio.h>
#fuses  XT, NOWDT, NOPROTECT, PUT

#use STANDARD_IO(A)
#use STANDARD_IO(B)
#use FIXED_IO(C_OUTPUTS=PIN_C0,PIN_C1,PIN_C2,PIN_C3)
#use DELAY(CLOCK=4000000)

static short nm, timeout;
static int   i, k, n, flag, dflag, count, count2, tbase;
static int   ptr, v_low, v_high, steppos, mflag;
static int   data2, data3, data4, array[16], array2[16];
static long  count3;

//********************************************************
//
//  To use a different remote control unit, change the data
//  in the following tables.
//
//*****

char steptable[4] = {5, 6, 10, 9};
char step_up[16] = {34,38,38,38,34,34,34,34,34,34,34,34,38,38,38,38};
char step_down[16] = {34,38,34,34,34,34,38,34,34,34,38,38,38,38,34,38};
char mute[16] = {34,38,38,38,38,34,34,34,34,34,34,34,34,38,38,38};

//*********************************************************
//
//  Get IR signal on and off pulse widths
//  1. Wait for low (setup in main or program)
//  2. Clear timer (in main or program)
//  3. Wait for high, read timer into v_low
//  4. Clear timer
//  5. Wait for low, read timer into v_high
//  6. Shift off high nibble of v_high and mask off high nibble
//     of v_low. Add and store in array
//
//  Routine makes two passes so it probably catches the last
//  16 bytes from the remote control, but not necessarily in time
//  sequence order.
//
//  1 timer tick for 4 MHz clock is 256 uS. 1/(4 Mhz/(4 * 256))
//
//*********************************************************

void measure()
       {
       SET_RTCC(0);	// clear timer
       count3 = 2500;	// long int
       timeout = 0;	// clear the flag bit
       count = 2;	// make 2 passes
me0:   n = 0;
me1:   if (input(pin_B7) == 1) goto me3;   // wait for high
       delay_US(50);
       if (input(pin_B7) == 1) goto me1;   // verify
       output_high(pin_B2);	// turn on carrier detect LED
       v_low = get_rtcc();                  // timer count
       SET_RTCC(0);                         // clear timer
me2:   if (input(pin_B7) == 0) goto me2;   // wait for low
       delay_US(50);
       if (input(pin_B7) == 0) goto me2;   // verify
       v_high = get_rtcc();
       if (v_high > 195) goto me9;	 // > 50 mS
       array[n] = (v_high << 4) + (v_low & 0x0F);
       SET_RTCC(0);                         // clear timer
       if (n < 15) {n += 1; goto me1;}
       count -= 1;
       if (count > 0) goto me0; else goto me9;
me3:   delay_US(200);	// return if no IR carrier for 500 mS
       count3 -= 1;
       if (count3 == 0) goto me9;
       goto me1;
me9:   return;
       }

//**********************************************************
//
// This routine compares a sequence of IR on/off times in "array"
// with data stored in the EEPROM (returned in data3).
// The byte at EEPROM addr zero is compared with the first byte
// in "array". If there is a match, the next bytes are compared
// until 16 bytes have been compared. If all match, "flag" is
// incremented to show which sequence matched (1, 2 or 3) and the
// routine returns. (flag = 0 indicates no match.)
// If any pair of bytes fail to match, "ptr" is reset to zero,
// EEPROM addr is set to the start of the next sequence and the
// comparison is repeated.
//
//************************************************************

void compare()
       {
       data2 = 0;                 // EEPROM start address
       ptr = 0; 	          // array pointer
       count3 = 16;	          // bytes in each sequence
       flag = 0;                  // match position if non-zero
cmp1:  data3 = read_eeprom(data2);      // read EEPROM at addr "data2"
					// puts byte in data3
       data4 = array[ptr];
       if (data3 == data4) goto cmp2;         // match
       goto cmp3;                             // no match
cmp2:  count3 -= 1;               // here if bytes match
       if (count3 == 0) {flag = 1; goto cmp9;}      // done
       data2 += 1;		// bump eeprom read addr
       ptr += 1;		// bump array pointer
       goto cmp1;		// compare next byte

cmp3:  data2 = 16;		// second eeprom start addr
       ptr = 0;			// reset array pointer
       count3 = 16;		// reset bytes in sequence
cmp4:  data3 = read_eeprom(data2);	// read eeprom
       data4 = array[ptr];
       if (data3 == data4) goto cmp5;
       goto cmp6;		// no matches
cmp5:  count3 -= 1;
       if (count3 == 0) {flag = 2; goto cmp9;}
       data2 += 1;		// bump eeprom read addr
       ptr += 1;		// bump array pointer
       goto cmp4;

cmp6:  data2 = 32;		// third eeprom start addr
       ptr = 0;
       count3 = 16;
cmp7:  data3 = read_eeprom(data2);
       data4 = array[ptr];
       if (data3 == data4) goto cmp8;
       goto cmp9;		// no match
cmp8:  count3 -= 1;
       if (count3 == 0) {flag = 3; goto cmp9;}
       data2 += 1;
       ptr += 1;
       goto cmp7;

cmp9:  return;
       }

//*********************************************************
//
//  Program the eeprom
//  Programs up and down data for TC101 Remote Control unit
//
//**********************************************************

void setup_remote()
       {
       n = 0;
       i = 0;
       k = 0;
set1:  data2 = step_up[n];
       write_eeprom(n, data2);
       delay_MS(10);
       n += 1;
       if (n <= 15) goto set1;
set2:  data2 = step_down[i];
       write_eeprom(n, data2);
       delay_MS(10);
       n += 1;
       i += 1;
       if (n <= 31) goto set2;
set3:  data2 = mute[k];
       write_eeprom(n, data2);
       delay_MS(10);
       n += 1;
       k += 1;
       if (n <= 47) goto set3;
       }

//***********************************************************
//
// This routine steps one step in the CW direction
//
//***********************************************************

void step_cw()
	{
	n = steppos;
        if (n == 3) n = 0; else n += 1;
        steppos = n;
        output_c(steptable[n]);
	delay_MS(20);
	output_c(0);
	delay_MS(20);
	k = 48;
	write_eeprom(k, steppos);
	}

//*********************************************************
//
// This routine steps one step in the CCW direction
//
//*********************************************************

void step_ccw()
	{
	n = steppos;
	if (n == 0) n = 3; else n -= 1;
	steppos = n;
	output_c(steptable[n]);
	delay_MS(20);
	output_c(0);
	delay_MS(20);
	k = 48;
	write_eeprom(k, steppos);
	}

//**********************************************************
//
//  This routine toggles the mute function, either on or off,
//  depending on the state of the mflag.
//
//**********************************************************

void check_mute()
       {
       if (mflag != 0) goto ckm2;	// jump if mute is on
       output_high(pin_B0);		// turn mute relay on
       output_high(pin_B1);		// turn on mute LED
       mflag = 1;			// set mflag
       goto ckm9;
ckm2:  output_low(pin_B0);		// turn relay off
       output_low(pin_B1);		// turn LED off
       mflag = 0;			// clear mflag
ckm9:  return;
       }

//*********************************************************
//
//  stepper motor initialization
//
//**********************************************************

void init_motor()
       {
       output_c(0);
       delay_MS(20);
       output_c(0);
       delay_MS(20);
       }

main()
       {
       mflag = 0;		// clear mute flag
       k = 48;
       steppos = read_eeprom(k);
       output_low(pin_B0);	// turn off mute relay
       output_low(pin_B1);	// turn off mute LED
       output_low(pin_B2);	// turn off IR detect LED
       setup_remote();
       init_motor();		// initialize stepper motor
       setup_counters(RTCC_internal,RTCC_DIV_256);
m1:    if (input(pin_B7) == 1) goto m1;  // wait for IR
       delay_US(50);
       if (input(pin_B7) == 1) goto m1;  // verify
       measure();
       delay_MS(200);
       compare();                 // compare to EEPROM
       if (flag == 0) goto m9;    // no match
       if (flag == 1) {step_cw(); goto m9;} 	  // step CW
       if (flag == 2) {step_ccw(); goto m9;}	 // step CCW
       if (flag == 3) check_mute();
m9:    output_low(pin_B2);	  // turn IR detect LED off
       delay_MS(200);
       goto m1;
       }
