
//    Ardunio based tube power amplifier controller
//    Mark Driedger           v2.0    January 6, 2015
//        
//    Automatically controls power on sequencing and power off for a tube amplifier 
//    based on sensing the presence of an analog audio input.  Provides calibration 
//    indication for output stage bias as well as over temperature shut down.
//  
//    When audio first appears, P0 goes high and enables power (filament supply
//    on tube amp).  After 10 seconds (filament warm up on tube amps), P1 goes high and 
//    enables the B+. When audio is absent for more than 45 seconds, then the amplifier
//    is turned off. If the ManualOff input is low, then the amplifier is turned off
//    regardless of the audio input or current power on sequencing state.  If the 
//    ManualOn input is low, then the amplifier is turned on regardless of the audio 
//    input, going through the same sequencing of P0, P1
//     
//    If both ManualOff and ManualOn are low, then the controller enters calibration 
//    mode and sequentially reads 4 ADC channels, averages them and controls 4 
//    tricolor LEDs to indicate if bias values are correct.  LEDs are multiplexed
//    with a 1/4 duty cycle.  If BiasMode port is low, then independent bias controls
//    are assumed for each tube.  Otherwise, bias/balance controls are assumed.
//     
//    Monitors an LM34 temperature sensor.  If temperature exceeds a
//    threshold then the amplifier shuts down until the temperature returns to normal
//    and the bias calibration LEDs are alternately flashed red at 2 Hz


#include  "AC.h"

//-------------------------------------------------------------------------------------
//    Modify the contants below to customize operation of the controller
//-------------------------------------------------------------------------------------
#define    TEMP_THRESHOLD      120.0    // over temperature shutdown threshold in F
#define    ON_DELAY            10       // filament warm up delay in seconds
#define    OFF_DELAY           45       // amplifier shut down delay in seconds
#define    BIAS_THRESHOLD      2.0      // bias error threshold in mA
#define    TARGET_BIAS         61.5     // target bias current in mA
#define    RS                  10.0     // output tube cathode sense resistor in ohms

void setup() 
//-------------------------------------------------------------------------------------
//  setup ports, disable calibration LEDs and turn off main/B+ power
//-------------------------------------------------------------------------------------
  { 
    pinMode(PortLEDred,    OUTPUT);     // configure I/O ports
    pinMode(PortLEDgrn,    OUTPUT);
    pinMode(PortLEDblu,    OUTPUT);
    pinMode(PortLEDL1,     OUTPUT); 
    pinMode(PortLEDL2,     OUTPUT);
    pinMode(PortLEDR1,     OUTPUT);
    pinMode(PortLEDR2,     OUTPUT);
    pinMode(PortP0,        OUTPUT); 
    pinMode(PortP1,        OUTPUT);
    pinMode(PortLoopTime,  OUTPUT);
    pinMode(PortLEDdebug,  OUTPUT);
    pinMode(PortManualOff, INPUT_PULLUP);
    pinMode(PortManualOn,  INPUT_PULLUP);
    pinMode(PortBiasMode,  INPUT_PULLUP);
    
    analogReference(INTERNAL);         // select 2.56V internal reference
    digitalWrite(PortLEDdebug, LOW);   // turn off debug (audio present) LED
    digitalWrite(PortP1, LOW);         // turn off main power
    digitalWrite(PortP0, LOW);         // ... and B+
    LEDDisable();                      // turn off calibration LEDs
  }

int LEDCal(float t, float v)
//-------------------------------------------------------------------------------------
//  compares v to threshold t and returns the appropriate LED color code
//  
//        -5t      -2.5t   -t      +t      +2.5t    +5t
//    blue  | blue  | green | green | green |  red  | red
//    flash |       | flash |       | flash |       | flash
//-------------------------------------------------------------------------------------

  {
    float   r = v/t;
    
    if (abs(r) <= 1.0)  return PortLEDgrn;
    if (abs(r) <= 2.5)  return PortLEDgrn | LEDFlash;
    if (r >  5.0)       return PortLEDred | LEDFlash;
    if (r >  2.5)       return PortLEDred;
    if (r < -5.0)       return PortLEDblu | LEDFlash;
    if (r < -2.5)       return PortLEDblu;
  }

void LEDDisable()
//-------------------------------------------------------------------------------------
//  disable all of the LED outputs (cathodes high, anodes low)
//-------------------------------------------------------------------------------------
  {
    digitalWrite(PortLEDred, HIGH);    // set cathodes high
    digitalWrite(PortLEDgrn, HIGH);
    digitalWrite(PortLEDblu, HIGH);

    digitalWrite(PortLEDL1,  LOW);     // and anodes low
    digitalWrite(PortLEDL2,  LOW);
    digitalWrite(PortLEDR1,  LOW);
    digitalWrite(PortLEDR2,  LOW);
  }  
  
void LEDwrite(int color, int led)
//--------------------------------------------------------------------------------------
//  enables the specified LED and color, turning all others off
//  - if the bit in color masked by LEDFlash is true, then LED is flashed based on modulo
//  division of the system timer at roughly 8 Hz
//--------------------------------------------------------------------------------------

  {
    const unsigned long  MASK_FLASH = 0x40;     // f = 1000/(0x40*2) = 7.8 Hz

    LEDDisable();                               // turn off LEDs
    if (!(color & LEDFlash) || (millis() & MASK_FLASH))
      {
        digitalWrite(color&0xf, LOW);    // set LED color (cathode low)
        digitalWrite(led,       HIGH);   // ... and enable the LED (anode high) 
      }
  }


void loop() 
//-------------------------------------------------------------------------------------
//  main processing loop
//  - update operating mode based on mode switches and temperature sensors
//  - update state machine
//  - if in calibration mode, then update calibration readings and LED settings
//-------------------------------------------------------------------------------------

  {
    static modeType  modeMap[2][2][2] = {{{ModeCAL,      ModeON},
                                          {ModeOFF,      ModeAUTO}}, 
                                         {{ModeOVERTEMP, ModeOVERTEMP}, 
                                          {ModeOVERTEMP, ModeOVERTEMP}}};
                                      
    boolean          ONswitch, OFFswitch, TempAlarm;
    const float      TempScale = 2.5/1024/0.01;     // 10mV/C into 10 bit, 2.5V FS
    enum modeType    mode;                          // control mode 
    enum stateType   CurrentState;                  // state machine state
    
    // read switches and temperature sensors to determine mode	 
    TempAlarm  = analogRead(PortTemp)*TempScale > TEMP_THRESHOLD;	  	
    ONswitch   = digitalRead(PortManualOn);
    OFFswitch  = digitalRead(PortManualOff);
    mode       = modeMap[TempAlarm][ONswitch][OFFswitch];
   
    CurrentState = UpdateStateMachine(mode);

    if (mode == ModeOVERTEMP)                // alternately flash bias LEDs red, 2 Hz
      {
        LEDwrite(PortLEDred, PortLEDL1);
        delay(250);
        LEDwrite(PortLEDred, PortLEDR1);
        delay(250);
      }
      
    if ((mode == ModeCAL) && (CurrentState == StateON))
      UpdateCalibration();  
  }
    
void UpdateCalibration()
//---------------------------------------------------------------------------------------
//  this code and the enclosing loop() function implements two activities
//  - each time through loop() it is called once and ...
//    - one of 4 ADC inputs are read
//    - one of 4 LEDs are lit (mulitiplexing), based on the ADC data
//  - LED status is updated every 4th call (after all 4 ADC channels are read)
//---------------------------------------------------------------------------------------
 
  { 
    // I/O and ADC port numbers corresponding to the arrays adc[] and LEDStatus[]
    const int       ADCMap[4] = {PortADCLA, PortADCLB, PortADCRA, PortADCRB};
    const int       LEDMap[4] = {PortLEDL1, PortLEDL2, PortLEDR1, PortLEDR2};                                 
    const float     ADC_FS    = 2560.0/RS;   // ADC full scale reading in mA
    const float     Io        = TARGET_BIAS; // desired output stage bias (per tube) in mA    
    const float     beta      = 0.875;       // filter constant for calibration measurements
    static int      LEDStatus[4];            // status of calibration LEDs
    static int      n = 0;                   // ADC and LED index
    
    static float    adc[4] = {0.0, 0.0, 0.0, 0.0};  // in mA

    adc[n] = beta*adc[n] + (1-beta) * analogRead(ADCMap[n])*ADC_FS/1024;    // read & filter  
 
    if (n==3)
        if (digitalRead(PortBiasMode))  // bias/balance mode
          {
            LEDStatus[0] = LEDCal(BIAS_THRESHOLD, (adc[0]+adc[1])/2 - Io);  // L1 = left bias
            LEDStatus[1] = LEDCal(BIAS_THRESHOLD,  adc[0]-adc[1]);          // L2 = left balance
            LEDStatus[2] = LEDCal(BIAS_THRESHOLD, (adc[2]+adc[3])/2 - Io);  // R1 = right bias
            LEDStatus[3] = LEDCal(BIAS_THRESHOLD,  adc[2]-adc[3]);          // R2 = right balance
          }
        else                            // independent bias mode
          {
            LEDStatus[0] = LEDCal(BIAS_THRESHOLD, adc[0] - Io);             // L1 = LA bias
            LEDStatus[1] = LEDCal(BIAS_THRESHOLD, adc[1] - Io);             // L2 = LB bias
            LEDStatus[2] = LEDCal(BIAS_THRESHOLD, adc[2] - Io);             // R1 = RA bias
            LEDStatus[3] = LEDCal(BIAS_THRESHOLD, adc[3] - Io);             // R2 = RB bias
          }      

      LEDwrite(LEDStatus[n], LEDMap[n]);  // set (multiplex) the next LED
      n = (n+1) % 4;                      // increment index and wrap
  }  
  
stateType UpdateStateMachine(modeType mode)
//--------------------------------------------------------------------------------------
//  measures the rectified analog inputs and implements the state machine
//  - mode is the operating mode as set by the power switches and temp sensors
//  - returns the current state
//  - as far as the state machine is concerned, ModeON and ModeCAL are identical
//  ... as are ModeOFF and ModeOVERTEMP
//--------------------------------------------------------------------------------------

  {
    const  int         ThA      = 25;       // threshold to declare audio present in ADC LSB's
    const  float       ThB      = 0.05;     // threshold to declare AudioPresent after filtering
    const  float       alpha    = 0.9375;   // filter constant for audio present signal
    const  long        Fs       = 2900L;    // sample frequency in samples/sec (measured @ D5 o/p)
    const  long        DelayOn  = ON_DELAY*Fs;  // 10 sec turn on delay to warm filaments
    const  long        DelayOff = OFF_DELAY*Fs; // 45 sec hold off when audio is lost   
    static stateType   state    = StateOFF; // current state, initialize to off
    static long        CounterOn;           // counts down from DelayOn to 0
    static long        CounterOff;          // counts down from DelayOff to 0
    static float       AudioFiltered = 0.0; // filtered audio present signal   
    boolean            AudioPresent;        // true if audio input is present
    int                AudioRaw;
    
    LEDDisable();                           // turn off calibration LEDs
    digitalWrite(PortLoopTime, HIGH);
		
    //read audio inputs, compare to threshold A, filter and then compare to threshold B  
    AudioRaw = analogRead(PortAudio) > ThA;
    AudioFiltered = alpha*AudioFiltered + (1-alpha)* AudioRaw;                
    AudioPresent  = AudioFiltered > ThB;
    digitalWrite(PortLEDdebug, AudioPresent);
    digitalWrite(PortLoopTime, LOW);
     
    if ((mode == ModeOFF) || (mode == ModeOVERTEMP))
      state = StateOFF; 
  	
    switch (state)                          // implement state machine
      {
        case StateOFF:
          digitalWrite(PortP1,  LOW);       // disable main power
          digitalWrite(PortP0,  LOW);       // disable B+
          if ((mode == ModeON) || (mode == ModeCAL) || ((mode == ModeAUTO) && AudioPresent))
            {
              state = StateDELAY;
              CounterOn = DelayOn;
              digitalWrite(PortP0, HIGH); // turn on main PSU (P0)
            }
            break;
        case StateDELAY:
          if (--CounterOn <= 0)            
            {
              state = StateON;               
              CounterOff = DelayOff;
              digitalWrite(PortP1, HIGH); // turn on B+/bias PSU (P1)
            }
          break;
        case StateON:
          if (mode == ModeAUTO)
            {
              if (AudioPresent) 
                CounterOff = DelayOff;    // re-trigger the timer
              else if (--CounterOff <=0) 
                state = StateOFF;
            }
          break;
      }
    return (state);
  }

